<?php
/**
 * WordPress LMS İstemcisi — PHP Referans Uygulaması
 *
 * Bu dosya WordPress eklentinize entegre edilecek istemci kodudur.
 * Tüm API istekleri HMAC-SHA256 ile imzalanır.
 *
 * Kurulum:
 *   1. Lisansı aktive ettikten sonra dönen secret_token'ı wp_options'a kaydedin.
 *   2. WP Cron ile günde bir kez lms_daily_verify() çağırın.
 *
 * @package YourPlugin\License
 */

class LmsClient
{
    private string $apiBase;
    private string $licenseKey;
    private string $secretToken;

    public function __construct(string $apiBase, string $licenseKey, string $secretToken)
    {
        $this->apiBase     = rtrim($apiBase, '/');
        $this->licenseKey  = $licenseKey;
        $this->secretToken = $secretToken;
    }

    // ─── Public API ────────────────────────────────────────────────────────────

    /** İlk aktivasyon — eklenti aktive edildiğinde çağrılır */
    public function activate(string $domain): array
    {
        return $this->request('POST', 'api/v1/license/activate', ['domain' => $domain]);
    }

    /** Günlük doğrulama */
    public function verify(string $domain): array
    {
        return $this->request('POST', 'api/v1/license/verify', ['domain' => $domain]);
    }

    /** Deaktivasyon — eklenti kaldırıldığında çağrılır */
    public function deactivate(string $domain): array
    {
        return $this->request('POST', 'api/v1/license/deactivate', ['domain' => $domain]);
    }

    // ─── Core Request ──────────────────────────────────────────────────────────

    private function request(string $method, string $path, array $body = []): array
    {
        $timestamp = (string) time();
        $nonce     = bin2hex(random_bytes(16)); // 32 karakter hex
        $jsonBody  = wp_json_encode($body);

        $signature = $this->sign($method, $path, $timestamp, $nonce, $jsonBody);

        $response = wp_remote_request(
            $this->apiBase . '/' . ltrim($path, '/'),
            [
                'method'  => $method,
                'timeout' => 15,
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept'       => 'application/json',
                    'X-License-Key' => $this->licenseKey,
                    'X-Signature'   => $signature,
                    'X-Timestamp'   => $timestamp,
                    'X-Nonce'       => $nonce,
                ],
                'body' => $jsonBody,
            ]
        );

        if (is_wp_error($response)) {
            return ['status' => 'error', 'message' => $response->get_error_message(), 'data' => null];
        }

        $decoded = json_decode(wp_remote_retrieve_body($response), true);

        if (!is_array($decoded)) {
            return ['status' => 'error', 'message' => 'Geçersiz API yanıtı.', 'data' => null];
        }

        return $decoded;
    }

    /**
     * HMAC-SHA256 imzası oluşturur.
     * Sunucu tarafındaki VerifyHmacSignature::buildPayload() ile birebir eşleşmeli.
     *
     * Payload formatı (her satır \n ile ayrılır):
     *   METHOD
     *   path/without/leading/slash
     *   timestamp
     *   nonce
     *   raw_json_body
     */
    private function sign(
        string $method,
        string $path,
        string $timestamp,
        string $nonce,
        string $body
    ): string {
        $payload = implode("\n", [
            strtoupper($method),
            ltrim($path, '/'),
            $timestamp,
            $nonce,
            $body,
        ]);

        return hash_hmac('sha256', $payload, $this->secretToken);
    }
}

// ─── WordPress Entegrasyon Örneği ──────────────────────────────────────────────

/**
 * Eklenti aktive edildiğinde çalışır.
 */
function lms_on_plugin_activation(): void
{
    $client = lms_get_client();
    if (!$client) {
        return;
    }

    $result = $client->activate(home_url());

    if (($result['status'] ?? '') === 'success') {
        // secret_token'ı güvenli kaydet (bir daha dönmez!)
        update_option('lms_secret_token', $result['data']['secret_token'], false);
        update_option('lms_license_status', 'active', false);
        update_option('lms_last_verified', time(), false);
    } else {
        // Admin notice göster
        set_transient('lms_activation_error', $result['message'] ?? 'Aktivasyon başarısız.', 60);
    }
}
register_activation_hook(__FILE__, 'lms_on_plugin_activation');

/**
 * Eklenti deaktive edildiğinde çalışır.
 */
function lms_on_plugin_deactivation(): void
{
    $client = lms_get_client();
    if (!$client) {
        return;
    }

    $client->deactivate(home_url());

    delete_option('lms_secret_token');
    delete_option('lms_license_status');
}
register_deactivation_hook(__FILE__, 'lms_on_plugin_deactivation');

/**
 * Günlük WP Cron görevi.
 */
function lms_daily_verify(): void
{
    $client = lms_get_client();
    if (!$client) {
        return;
    }

    $result = $client->verify(home_url());
    $status = $result['data']['license_status'] ?? 'unknown';

    update_option('lms_license_status', $status, false);
    update_option('lms_last_verified', time(), false);

    // Admin'e gösterilecek mesaj varsa kaydet
    if (!empty($result['data']['admin_message'])) {
        update_option('lms_admin_message', $result['data']['admin_message'], false);
    }
}
add_action('lms_daily_cron', 'lms_daily_verify');

// Cron zamanla
if (!wp_next_scheduled('lms_daily_cron')) {
    wp_schedule_event(time(), 'daily', 'lms_daily_cron');
}

// ─── Yardımcılar ───────────────────────────────────────────────────────────────

function lms_get_client(): ?LmsClient
{
    $licenseKey  = get_option('lms_license_key');
    $secretToken = get_option('lms_secret_token');

    if (!$licenseKey || !$secretToken) {
        return null;
    }

    return new LmsClient(
        apiBase:     defined('LMS_API_URL') ? LMS_API_URL : 'https://your-lms-server.com',
        licenseKey:  $licenseKey,
        secretToken: $secretToken,
    );
}
