<?php
/**
 * Verdent License Protector
 *
 * WordPress eklentisi için kırılmaya karşı dirençli lisans koruma katmanı.
 * PHP 8.0+ uyumlu — Türkçe açıklamalı.
 *
 * Özellikler:
 *   - HMAC-SHA256 el sıkışma (replay attack korumalı)
 *   - AES-256-CBC şifreli wp_options depolama
 *   - 3 günlük mühlet (grace period) sistemi
 *   - Kill-Switch (uzaktan iptal + yerel temizlik)
 *   - Decoy (yem) mantığı — gerçek kontrol gizlendi
 *   - Admin mesaj sistemi (notice-warning)
 *   - WP Cron ile günlük sessiz doğrulama
 *
 * @package Verdent
 * @version 3.0.0
 */

declare(strict_types=1);

// Doğrudan erişim engeli
if (!defined('ABSPATH')) {
    exit;
}

// ─── Sabitler ────────────────────────────────────────────────────────────────

/** wp_options anahtarı — kasıtlı belirsiz isim */
define('VLP_STORE_KEY',   '_vrd_cfg_data');

/** Cron kanca adı */
define('VLP_CRON_HOOK',   'vrd_heartbeat_tick');

/** Mühlet süresi (saniye) — 3 gün */
define('VLP_GRACE_TTL',   60 * 60 * 24 * 3);

/** API zaman aşımı (saniye) */
define('VLP_API_TIMEOUT', 15);

/** Şifreleme algoritması */
define('VLP_CIPHER',      'AES-256-CBC');

// ─── Ana Sınıf ────────────────────────────────────────────────────────────────

/**
 * Class Verdent_License_Protector
 *
 * İsimlendirme notu: Sınıf ve metod adları kasıtlı olarak lisans yönetimine
 * atıfta bulunmaz. Obfuscation sonrası bu isimler tamamen değiştirilecektir.
 */
final class Verdent_License_Protector
{
    // ─── Özellikler ──────────────────────────────────────────────────────────

    /** API sunucu adresi */
    private readonly string $endpoint;

    /** Lisans anahtarı — wp_options'dan yüklenir */
    private string $lk = '';

    /** HMAC imzalama tokeni — şifreli depodan okunur, bellekte tutulur */
    private string $st = '';

    /** Singleton örneği */
    private static ?self $instance = null;

    /**
     * Gerçek iş mantığı sağlayıcısı.
     * Lisans kontrolü bu bağımlılığa enjekte edilir —
     * dışarıdan bakıldığında başka bir servis gibi görünür.
     */
    private Verdent_Content_Renderer $renderer;

    // ─── Başlatma ─────────────────────────────────────────────────────────────

    /**
     * Kurucu (private) — Singleton kalıbı.
     *
     * @param string $api_base   Laravel API sunucusunun kök URL'i
     * @param string $license_key Lisans anahtarı (XXXX-XXXX-XXXX-XXXX)
     */
    private function __construct(string $api_base, string $license_key)
    {
        $this->endpoint = rtrim($api_base, '/');
        $this->lk       = sanitize_text_field($license_key);

        // Şifreli depodan secret_token ve durum bilgisini yükle
        $this->_hydrate_runtime_context();

        // Bağımlılık: gerçek iş motoru burada enjekte edilir
        // Lisans durumu bu nesnenin kurucusuna geçirilir — dışarıdan görünmez
        $this->renderer = new Verdent_Content_Renderer(
            $this->_resolve_operational_state()
        );
    }

    /**
     * Singleton erişim noktası.
     *
     * @param string $api_base
     * @param string $license_key
     */
    public static function ignite(string $api_base, string $license_key): self
    {
        if (self::$instance === null) {
            self::$instance = new self($api_base, $license_key);
        }
        return self::$instance;
    }

    /** Clone engeli */
    private function __clone() {}

    /** Unserialize engeli */
    public function __wakeup(): never
    {
        throw new \RuntimeException('Serializasyon yasak.');
    }

    // ─── WordPress Entegrasyonu ───────────────────────────────────────────────

    /**
     * WordPress hook'larını bağlar.
     * Eklentinin ana plugin dosyasından çağrılır.
     *
     * Örnek kullanım (plugin.php):
     *   add_action('plugins_loaded', fn() =>
     *       Verdent_License_Protector::ignite(VRD_API_URL, get_option('vrd_lk'))
     *           ->mount()
     *   );
     */
    public function mount(): void
    {
        // Cron planla
        add_action('init', [$this, '_schedule_heartbeat']);
        add_action(VLP_CRON_HOOK, [$this, '_run_background_verification']);

        // Admin bildirimleri
        add_action('admin_notices', [$this, '_render_admin_notices']);

        // Ayarlar sayfası (lisans giriş formu)
        add_action('admin_menu', [$this, '_register_settings_page']);
        add_action('admin_post_vrd_save_license', [$this, '_handle_license_form']);

        // ─── Eklentinin ana işlevleri buraya bağlanır ─────────────────────────
        // Gerçek hook'lar renderer üzerinden koşullu yüklenir.
        // Lisans geçersizse renderer hiçbir çıktı üretmez.
        $this->renderer->register_hooks();
    }

    // ─── Cron ─────────────────────────────────────────────────────────────────

    /**
     * WP Cron planlamasını kurar.
     * Günde bir kez sessizce arka planda çalışır.
     */
    public function _schedule_heartbeat(): void
    {
        if (!wp_next_scheduled(VLP_CRON_HOOK)) {
            // Rastgele ofset: herkeste aynı saatte çalışmasın (sunucu yükünü dağıt)
            $offset = wp_rand(0, 3600);
            wp_schedule_event(time() + $offset, 'daily', VLP_CRON_HOOK);
        }
    }

    /**
     * Cron tetiklendiğinde çalışır — arka planda sessiz doğrulama.
     */
    public function _run_background_verification(): void
    {
        $this->_perform_verification_cycle(silent: true);
    }

    // ─── Doğrulama Döngüsü ───────────────────────────────────────────────────

    /**
     * API'ye HMAC imzalı istek atar ve sonucu işler.
     *
     * @param bool $silent  true = hata bildirimi gösterme (cron modu)
     * @return bool         Lisans geçerliyse true
     */
    private function _perform_verification_cycle(bool $silent = false): bool
    {
        // secret_token yoksa aktivasyon yapılmamış
        if (empty($this->st)) {
            return false;
        }

        $path      = 'api/v1/license/verify';
        $body      = wp_json_encode([
            'domain' => $this->_normalize_site_url(),
        ]);
        $headers   = $this->_build_hmac_headers('POST', $path, $body);

        $response  = wp_remote_post($this->endpoint . '/' . $path, [
            'timeout' => VLP_API_TIMEOUT,
            'headers' => $headers,
            'body'    => $body,
        ]);

        // ── Ağ Hatası / Sunucu Çökmesi → Mühlet Sistemi ──────────────────────
        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) >= 500) {
            return $this->_apply_grace_period($silent);
        }

        $code    = (int) wp_remote_retrieve_response_code($response);
        $payload = json_decode(wp_remote_retrieve_body($response), true);

        // ── Kill-Switch: Askıya alınmış / iptal edilmiş ───────────────────────
        $license_status = $payload['data']['license_status'] ?? $payload['license_status'] ?? '';
        if (in_array($license_status, ['suspended', 'revoked'], true) || $code === 403) {
            $this->_execute_kill_switch($payload);
            return false;
        }

        // ── Süresi Dolmuş ─────────────────────────────────────────────────────
        if ($license_status === 'expired' || $code === 402) {
            $this->_store_state([
                'status'      => 'expired',
                'verified_at' => time(),
                'message'     => $payload['data']['admin_message'] ?? null,
            ]);
            return false;
        }

        // ── Başarılı Doğrulama ────────────────────────────────────────────────
        if ($code === 200 && ($payload['status'] ?? '') === 'success') {
            $this->_store_state([
                'status'      => 'ok',                // 'active' yerine gizli değer
                'verified_at' => time(),
                'valid_until' => $payload['data']['valid_until'] ?? null,
                'message'     => $payload['data']['admin_message'] ?? null,
            ]);
            // Mühlet sayacını sıfırla
            delete_option('_vrd_gc_ts');
            return true;
        }

        return false;
    }

    // ─── 1) HMAC El Sıkışma ──────────────────────────────────────────────────

    /**
     * HMAC-SHA256 imzalı HTTP başlıklarını oluşturur.
     *
     * Payload şeması (sunucu ile birebir eşleşmeli):
     *   METHOD\npath\ntimestamp\nnonce\nbody
     *
     * @param string $method  HTTP metodu (POST)
     * @param string $path    API yolu (başında / olmadan)
     * @param string $body    JSON body
     * @return array          WordPress wp_remote_* uyumlu headers dizisi
     */
    private function _build_hmac_headers(string $method, string $path, string $body): array
    {
        $timestamp = (string) time();
        $nonce     = bin2hex(random_bytes(16)); // 32 karakter — kriptografik rastgelelik

        $payload   = implode("\n", [
            strtoupper($method),
            ltrim($path, '/'),
            $timestamp,
            $nonce,
            $body,
        ]);

        $signature = hash_hmac('sha256', $payload, $this->st);

        return [
            'Content-Type'  => 'application/json',
            'Accept'        => 'application/json',
            'X-License-Key' => $this->lk,
            'X-Signature'   => $signature,
            'X-Timestamp'   => $timestamp,
            'X-Nonce'       => $nonce,
        ];
    }

    // ─── 2) Şifreli Depolama (AES-256-CBC) ───────────────────────────────────

    /**
     * Durumu şifreleyerek wp_options'a kaydeder.
     *
     * Şifreleme anahtarı: WordPress AUTH_KEY sabiti (her site kendine özgü).
     * IV: Her şifrelemede rastgele üretilir, şifreli veriye ön ek olarak eklenir.
     *
     * Veritabanına kaydedilen değer örneği:
     *   "8f3a...iv_hex...:base64_encrypted_data"
     *
     * @param array $state  Kaydedilecek durum verisi
     */
    private function _store_state(array $state): void
    {
        $json      = wp_json_encode($state);
        $encrypted = $this->_aes_encrypt($json);

        if ($encrypted !== false) {
            update_option(VLP_STORE_KEY, $encrypted, 'no');
        }
    }

    /**
     * wp_options'tan şifreli durumu okur ve çözer.
     *
     * @return array|null  Çözülmüş durum verisi; yoksa null
     */
    private function _load_state(): ?array
    {
        $raw = get_option(VLP_STORE_KEY, null);
        if (!$raw) {
            return null;
        }

        $decrypted = $this->_aes_decrypt($raw);
        if ($decrypted === false) {
            return null;
        }

        $data = json_decode($decrypted, true);
        return is_array($data) ? $data : null;
    }

    /**
     * AES-256-CBC şifreleme.
     *
     * @param string $plaintext  Düz metin
     * @return string|false      "iv_hex:base64_ciphertext" veya hata durumunda false
     */
    private function _aes_encrypt(string $plaintext): string|false
    {
        $key = $this->_derive_encryption_key();
        $iv  = random_bytes(openssl_cipher_iv_length(VLP_CIPHER));

        $ciphertext = openssl_encrypt($plaintext, VLP_CIPHER, $key, OPENSSL_RAW_DATA, $iv);
        if ($ciphertext === false) {
            return false;
        }

        // IV + şifreli veri birlikte saklanır
        return bin2hex($iv) . ':' . base64_encode($ciphertext);
    }

    /**
     * AES-256-CBC şifre çözme.
     *
     * @param string $encrypted  "iv_hex:base64_ciphertext"
     * @return string|false      Düz metin veya hata durumunda false
     */
    private function _aes_decrypt(string $encrypted): string|false
    {
        $parts = explode(':', $encrypted, 2);
        if (count($parts) !== 2) {
            return false;
        }

        [$iv_hex, $ciphertext_b64] = $parts;

        $key        = $this->_derive_encryption_key();
        $iv         = hex2bin($iv_hex);
        $ciphertext = base64_decode($ciphertext_b64);

        if ($iv === false || $ciphertext === false) {
            return false;
        }

        return openssl_decrypt($ciphertext, VLP_CIPHER, $key, OPENSSL_RAW_DATA, $iv);
    }

    /**
     * WordPress AUTH_KEY sabitinden 32 byte'lık türetilmiş şifreleme anahtarı üretir.
     * HKDF (HMAC-based Key Derivation) — her site kendine özgü anahtar kullanır.
     *
     * @return string  32 byte binary anahtar
     */
    private function _derive_encryption_key(): string
    {
        $salt = defined('AUTH_KEY') ? AUTH_KEY : wp_salt('auth');
        return hash_hkdf('sha256', $salt, 32, 'verdent-license-v3', $this->lk);
    }

    // ─── 3) Mühlet Sistemi ────────────────────────────────────────────────────

    /**
     * API erişilemez olduğunda mühlet mantığını uygular.
     *
     * - İlk başarısızlıkta mühlet sayacını başlatır
     * - 3 gün dolmamışsa mevcut durumu geçerli kabul eder
     * - 3 gün dolmuşsa kısıtlı moda geçer
     *
     * @param bool $silent  Bildirim gösterilsin mi?
     * @return bool         Mühlet içindeyse true (eklenti çalışmaya devam eder)
     */
    private function _apply_grace_period(bool $silent): bool
    {
        $grace_start = (int) get_option('_vrd_gc_ts', 0);

        if ($grace_start === 0) {
            // İlk başarısızlık — mühlet sayacını başlat
            update_option('_vrd_gc_ts', time(), 'no');
            $this->_set_notice(
                'warning',
                'Lisans doğrulama sunucusuna şu an ulaşılamıyor. '
                . '3 günlük mühlet başlatıldı. Eklenti normal çalışmaya devam ediyor.'
            );
            return true;
        }

        $elapsed = time() - $grace_start;

        if ($elapsed < VLP_GRACE_TTL) {
            // Mühlet sürüyor — devam et
            $remaining_hours = (int) ceil((VLP_GRACE_TTL - $elapsed) / 3600);
            if (!$silent) {
                $this->_set_notice(
                    'warning',
                    "Lisans sunucusuna ulaşılamıyor. Mühlet: ~{$remaining_hours} saat kaldı."
                );
            }
            return true;
        }

        // Mühlet bitti — kısıtlı mod
        $this->_set_notice(
            'error',
            'Lisans doğrulaması 3 gündür yapılamadı. Eklenti kısıtlı moda geçti. '
            . 'Lütfen internet bağlantınızı kontrol edin.'
        );
        $this->_store_state(['status' => 'grace_expired', 'verified_at' => time()]);
        return false;
    }

    // ─── 4) Kill-Switch ───────────────────────────────────────────────────────

    /**
     * Sunucudan askıya alma / iptal sinyali geldiğinde tetiklenir.
     *
     * 1. Yerel şifreli depoyu tamamen temizler
     * 2. Admin panelinde kırmızı uyarı gösterir
     * 3. Renderer'ı kapatır (eklenti hook'ları devre dışı)
     *
     * @param array $payload  API yanıt payload'ı
     */
    private function _execute_kill_switch(array $payload): void
    {
        // Tüm yerel lisans verisini sil
        delete_option(VLP_STORE_KEY);
        delete_option('_vrd_gc_ts');
        delete_option('_vrd_ak_data');   // Aktivasyon verisi
        $this->st = '';

        // Kırmızı uyarı bildirimi
        $message = $payload['message']
            ?? $payload['data']['admin_message']
            ?? 'Lisansınız sunucu tarafından iptal edilmiştir. Lütfen destek ile iletişime geçin.';

        $this->_set_notice('error', '🚫 ' . $message);

        // Cron'u kaldır
        wp_clear_scheduled_hook(VLP_CRON_HOOK);

        // Renderer'ı kapat — tüm eklenti hook'larını devre dışı bırakır
        $this->renderer->shutdown();

        // Admin'e kalıcı uyarı kaydet (sayfa yenilensede görünsün)
        update_option('_vrd_kill_notice', sanitize_text_field($message), 'no');
    }

    // ─── 5) Decoy (Yem) Mantığı ──────────────────────────────────────────────

    /**
     * Gerçek lisans durumunu belirler.
     *
     * Bu metod kasıtlı olarak "operasyonel durum çözücü" gibi adlandırıldı.
     * Lisans kontrolü dışarıdan bağımlılık olarak enjekte edilir;
     * `is_license_active()` gibi aranabilir bir metod yoktur.
     *
     * @return bool  Eklenti tam işlevli çalışabilir mi?
     */
    private function _resolve_operational_state(): bool
    {
        // Kill-switch kalıcı kaydı var mı?
        if (get_option('_vrd_kill_notice')) {
            return false;
        }

        $state = $this->_load_state();

        if (!$state) {
            return false;
        }

        // Durum kontrolü — 'active' veya 'true' yerine gizli değer kullanılıyor
        if (($state['status'] ?? '') !== 'ok') {
            return false;
        }

        // Doğrulama zamanı kontrolü — çok eskiyse cron atlandı demektir
        $verified_at = (int) ($state['verified_at'] ?? 0);
        if (time() - $verified_at > (VLP_GRACE_TTL * 2)) {
            // 6 günden fazla doğrulama yoksa kısıtla
            return false;
        }

        return true;
    }

    /**
     * Yem metodu.
     * Crack'çının doğrudan bypass etmeye çalışacağı sahte fonksiyon.
     * Bu metodun döndürdüğü değer hiçbir yerde kullanılmaz.
     *
     * @deprecated Kullanılmaz — sadece yanıltma amaçlıdır.
     * @return bool  Her zaman true döner (değersiz)
     */
    public function is_license_active(): bool  // phpcs:ignore
    {
        // Bu fonksiyonun dönüş değeri sistemde kullanılmıyor.
        // Gerçek kontrol _resolve_operational_state() içinde.
        return true;
    }

    // ─── 6) Admin Mesajları ───────────────────────────────────────────────────

    /**
     * WordPress admin panelinde bildirimleri render eder.
     * Yalnızca eklentinin kendi ayarlar sayfasında gösterilir.
     */
    public function _render_admin_notices(): void
    {
        // Sadece kendi ayarlar sayfamızda göster
        $screen = get_current_screen();
        if (!$screen || !str_contains($screen->id, 'vrd-license')) {
            // Kill-switch bildirimi tüm admin sayfalarında görünür
            $kill = get_option('_vrd_kill_notice');
            if ($kill) {
                echo '<div class="notice notice-error is-dismissible">'
                    . '<p><strong>Verdent Lisans:</strong> '
                    . esc_html($kill) . '</p></div>';
            }
            return;
        }

        // Birikmiş bildirimleri göster
        $notices = get_transient('_vrd_notices') ?: [];
        foreach ($notices as $notice) {
            $type    = esc_attr($notice['type']);
            $message = esc_html($notice['message']);
            echo "<div class=\"notice notice-{$type} is-dismissible\"><p><strong>Verdent:</strong> {$message}</p></div>";
        }
        delete_transient('_vrd_notices');

        // API'den gelen admin_message — yalnızca ayarlar sayfasında
        $state = $this->_load_state();
        if (!empty($state['message'])) {
            echo '<div class="notice notice-warning"><p>'
                . '<strong>📢 Lisans Bildirimi:</strong> '
                . esc_html($state['message']) . '</p></div>';
        }
    }

    /**
     * Bildirim kuyruğuna ekler.
     *
     * @param string $type     'error' | 'warning' | 'success' | 'info'
     * @param string $message  Bildirim metni
     */
    private function _set_notice(string $type, string $message): void
    {
        $notices   = get_transient('_vrd_notices') ?: [];
        $notices[] = ['type' => $type, 'message' => $message];
        set_transient('_vrd_notices', $notices, HOUR_IN_SECONDS);
    }

    // ─── Aktivasyon ───────────────────────────────────────────────────────────

    /**
     * İlk aktivasyonu gerçekleştirir.
     * Eklenti etkinleştirildiğinde çağrılır (register_activation_hook).
     *
     * @return bool  Başarılıysa true
     */
    public function activate_license(): bool
    {
        $path = 'api/v1/license/activate';
        $body = wp_json_encode(['domain' => $this->_normalize_site_url()]);

        // Aktivasyon için henüz secret_token yok;
        // geçici olarak lisans anahtarından türetilmiş bir ön-imza kullan
        // (Sunucu bu aşamada HMAC doğrulamasını lisans kaydından yapar)
        $headers = $this->_build_hmac_headers('POST', $path, $body);

        $response = wp_remote_post($this->endpoint . '/' . $path, [
            'timeout' => VLP_API_TIMEOUT,
            'headers' => $headers,
            'body'    => $body,
        ]);

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 201) {
            $msg = is_wp_error($response)
                ? $response->get_error_message()
                : (json_decode(wp_remote_retrieve_body($response), true)['message'] ?? 'Sunucu hatası');
            $this->_set_notice('error', 'Aktivasyon başarısız: ' . $msg);
            return false;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        // Secret token'ı şifreli depola (bir daha API'den gelmez!)
        $this->st = $data['data']['secret_token'];
        $this->_store_state([
            'status'      => 'ok',
            'verified_at' => time(),
            'valid_until' => $data['data']['valid_until'] ?? null,
            'message'     => null,
        ]);

        // secret_token'ı ayrı şifreli anahtar altında da sakla
        $this->_store_secret_token($data['data']['secret_token']);

        $this->_set_notice('success', 'Lisans başarıyla aktive edildi! 🎉');
        return true;
    }

    /**
     * Deaktivasyon — eklenti kaldırıldığında çağrılır.
     */
    public function deactivate_license(): void
    {
        if (empty($this->st)) {
            return;
        }

        $path = 'api/v1/license/deactivate';
        $body = wp_json_encode(['domain' => $this->_normalize_site_url()]);

        wp_remote_post($this->endpoint . '/' . $path, [
            'timeout'   => VLP_API_TIMEOUT,
            'headers'   => $this->_build_hmac_headers('POST', $path, $body),
            'body'      => $body,
            'blocking'  => false, // Eklenti kaldırılırken yanıt bekleme
        ]);

        delete_option(VLP_STORE_KEY);
        delete_option('_vrd_ak_data');
        delete_option('_vrd_gc_ts');
        wp_clear_scheduled_hook(VLP_CRON_HOOK);
    }

    // ─── Ayarlar Sayfası ─────────────────────────────────────────────────────

    /**
     * Eklenti ayarlar sayfasını admin menüsüne ekler.
     */
    public function _register_settings_page(): void
    {
        add_options_page(
            page_title: 'Verdent Lisans',
            menu_title: 'Verdent Lisans',
            capability: 'manage_options',
            menu_slug:  'vrd-license',
            callback:   [$this, '_render_settings_page'],
        );
    }

    /**
     * Lisans ayarlar sayfasını render eder.
     */
    public function _render_settings_page(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        $state       = $this->_load_state();
        $status      = $state['status'] ?? 'unknown';
        $verified_at = $state['verified_at'] ?? null;
        $valid_until = $state['valid_until'] ?? null;

        $status_label = match ($status) {
            'ok'            => '<span style="color:#46b450">✅ Aktif</span>',
            'expired'       => '<span style="color:#dc3232">⏰ Süresi Doldu</span>',
            'grace_expired' => '<span style="color:#ffb900">⚠️ Kısıtlı Mod</span>',
            default         => '<span style="color:#dc3232">❌ Doğrulanmamış</span>',
        };
        ?>
        <div class="wrap">
            <h1>🔑 Verdent Lisans Yönetimi</h1>

            <table class="form-table" role="presentation">
                <tr>
                    <th>Lisans Anahtarı</th>
                    <td><code><?php echo esc_html($this->lk ?: '—'); ?></code></td>
                </tr>
                <tr>
                    <th>Durum</th>
                    <td><?php echo $status_label; ?></td>
                </tr>
                <?php if ($verified_at): ?>
                <tr>
                    <th>Son Doğrulama</th>
                    <td><?php echo esc_html(wp_date('d.m.Y H:i', $verified_at)); ?></td>
                </tr>
                <?php endif; ?>
                <?php if ($valid_until): ?>
                <tr>
                    <th>Bitiş Tarihi</th>
                    <td><?php echo esc_html(wp_date('d.m.Y H:i', strtotime($valid_until))); ?></td>
                </tr>
                <?php endif; ?>
            </table>

            <hr>

            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <input type="hidden" name="action" value="vrd_save_license">
                <?php wp_nonce_field('vrd_license_nonce', '_vrd_n'); ?>

                <table class="form-table">
                    <tr>
                        <th><label for="vrd_lk">Lisans Anahtarı</label></th>
                        <td>
                            <input type="text"
                                   id="vrd_lk"
                                   name="vrd_lk"
                                   value="<?php echo esc_attr($this->lk); ?>"
                                   class="regular-text code"
                                   placeholder="XXXX-XXXX-XXXX-XXXX">
                        </td>
                    </tr>
                </table>

                <?php submit_button('Lisansı Kaydet & Aktive Et'); ?>
            </form>

            <?php if ($status === 'ok'): ?>
                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin-top:10px">
                    <input type="hidden" name="action" value="vrd_save_license">
                    <input type="hidden" name="vrd_action" value="manual_verify">
                    <?php wp_nonce_field('vrd_license_nonce', '_vrd_n'); ?>
                    <?php submit_button('Manuel Doğrulama Yap', 'secondary'); ?>
                </form>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Lisans form gönderimini işler.
     */
    public function _handle_license_form(): void
    {
        if (!current_user_can('manage_options')
            || !check_admin_referer('vrd_license_nonce', '_vrd_n')) {
            wp_die('Yetkisiz erişim.', 403);
        }

        $action = sanitize_key($_POST['vrd_action'] ?? 'save');

        if ($action === 'manual_verify') {
            $result = $this->_perform_verification_cycle(silent: false);
            $this->_set_notice(
                $result ? 'success' : 'error',
                $result ? 'Doğrulama başarılı.' : 'Doğrulama başarısız.'
            );
        } else {
            $new_key   = sanitize_text_field($_POST['vrd_lk'] ?? '');
            if (empty($new_key)) {
                wp_safe_redirect(admin_url('options-general.php?page=vrd-license'));
                exit;
            }
            update_option('vrd_lk', $new_key, 'no');
            $this->lk  = $new_key;
            $this->activate_license();
        }

        wp_safe_redirect(admin_url('options-general.php?page=vrd-license'));
        exit;
    }

    // ─── Yardımcı Metodlar ────────────────────────────────────────────────────

    /**
     * Secret token'ı ayrı bir şifreli anahtar altında saklar.
     * Hydration sırasında buradan okunur.
     */
    private function _store_secret_token(string $token): void
    {
        $encrypted = $this->_aes_encrypt($token);
        if ($encrypted !== false) {
            update_option('_vrd_ak_data', $encrypted, 'no');
        }
    }

    /**
     * Runtime başlangıcında şifreli depodan secret_token ve durumu yükler.
     */
    private function _hydrate_runtime_context(): void
    {
        // Secret token'ı çöz
        $raw_token = get_option('_vrd_ak_data', null);
        if ($raw_token) {
            $decrypted = $this->_aes_decrypt($raw_token);
            if ($decrypted !== false) {
                $this->st = $decrypted;
            }
        }
    }

    /**
     * Site URL'ini normalize eder (www., protokol, trailing slash kaldırır).
     */
    private function _normalize_site_url(): string
    {
        $url = get_site_url();
        $url = preg_replace('#^https?://#', '', $url);
        $url = ltrim($url, 'www.');
        return rtrim($url, '/');
    }
}

// =============================================================================
// Verdent_Content_Renderer — İşlevsel Bağımlılık
// =============================================================================

/**
 * Eklentinin asıl işlevlerini yürüten sınıf.
 * Lisans durumu constructor'da DI ile enjekte edilir.
 *
 * Bu sınıf; bir lisans sınıfı değil, "içerik render motorudur".
 * Crack'çı için yanıltıcı bir hedef.
 */
final class Verdent_Content_Renderer
{
    /**
     * @param bool $is_operational  Enjekte edilen lisans durumu
     */
    public function __construct(
        private readonly bool $is_operational
    ) {}

    /**
     * Eklentinin WordPress hook'larını kaydeder.
     * Lisans geçersizse hiçbir hook bağlanmaz.
     */
    public function register_hooks(): void
    {
        if (!$this->is_operational) {
            // Kısıtlı modda sadece ayarlar sayfası için gerekli minimum hook'lar
            return;
        }

        // ── EKLENTININ GERÇEK HOOK'LARI BURAYA GELİR ─────────────────────────
        // Örnek:
        // add_filter('the_content', [$this, 'process_content']);
        // add_shortcode('verdent', [$this, 'render_shortcode']);
        // add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
        // ─────────────────────────────────────────────────────────────────────
    }

    /**
     * Kill-Switch çalıştığında tüm hook'ları kaldırır.
     */
    public function shutdown(): void
    {
        remove_all_actions('verdent_render');
        remove_all_filters('the_content');
        // Eklentiye özel tüm hook'lar buraya eklenmeli
    }
}
