<?php

namespace App\Services;

use App\Models\License;
use App\Models\LicenseActivationLog;
use Illuminate\Support\Facades\DB;
use RuntimeException;

/**
 * Lisans iş mantığı servisi.
 *
 * Veritabanı yazma operasyonları doğrudan PDO prepared statements
 * ile gerçekleştirilir — SQL injection'a karşı maksimum güvence.
 *
 * Okuma operasyonları Eloquent ile yapılır (zaten parametre binding kullanır).
 */
class LicenseService
{
    // ─── Aktivasyon ────────────────────────────────────────────────────────────

    /**
     * İlk aktivasyonu gerçekleştirir:
     *  1. Lisans aktif mi?
     *  2. Slot var mı?
     *  3. Domain kilitli mi? (zaten başka bir domain'e mi bağlı?)
     *  4. Atomic UPDATE ile domain kilitle + activations++
     *  5. Log yaz
     *
     * @return array{license: License, secret_token: string}
     * @throws RuntimeException
     */
    public function activate(License $license, string $domain, string $ip, string $userAgent): array
    {
        // --- 1. Durum kontrolleri ---
        if ($license->isSuspended()) {
            $this->writeLog($license->id, 'failed', $domain, $ip, $userAgent, 'Lisans askıya alınmış.');
            throw new RuntimeException('SUSPENDED');
        }

        if ($license->isExpired()) {
            $this->writeLog($license->id, 'failed', $domain, $ip, $userAgent, 'Lisansın süresi dolmuş.');
            throw new RuntimeException('EXPIRED');
        }

        if (!$license->isActive()) {
            $this->writeLog($license->id, 'failed', $domain, $ip, $userAgent, 'Lisans aktif değil.');
            throw new RuntimeException('INACTIVE');
        }

        // --- 2. Domain çakışma kontrolü ---
        if ($license->isDomainLocked() && !$license->matchesDomain($domain)) {
            $this->writeLog(
                $license->id, 'failed', $domain, $ip, $userAgent,
                "Domain uyumsuzluğu: kayıtlı={$license->domain}, gelen={$domain}"
            );
            throw new RuntimeException('DOMAIN_MISMATCH');
        }

        // --- 3. Slot kontrolü (domain zaten kilitliyse tekrar aktivasyon sayılmaz) ---
        $isReactivation = $license->isDomainLocked() && $license->matchesDomain($domain);

        if (!$isReactivation && !$license->hasActivationSlot()) {
            $this->writeLog(
                $license->id, 'failed', $domain, $ip, $userAgent,
                "Aktivasyon limiti aşıldı: {$license->current_activations}/{$license->activation_limit}"
            );
            throw new RuntimeException('ACTIVATION_LIMIT_REACHED');
        }

        // --- 4. Atomic PDO UPDATE — race condition koruması ---
        DB::transaction(function () use ($license, $domain, $isReactivation) {
            $pdo = DB::getPdo();

            if ($isReactivation) {
                // Domain zaten bu lisansa kilitli; sadece last_verify_at güncelle
                $stmt = $pdo->prepare(
                    'UPDATE licenses
                        SET last_verify_at = NOW(),
                            updated_at     = NOW()
                      WHERE id = :id
                        AND status = :status
                        AND deleted_at IS NULL'
                );
                $stmt->execute([':id' => $license->id, ':status' => 'active']);
            } else {
                // Yeni aktivasyon: domain kilitle, sayacı artır
                $stmt = $pdo->prepare(
                    'UPDATE licenses
                        SET domain               = :domain,
                            current_activations  = current_activations + 1,
                            last_verify_at       = NOW(),
                            updated_at           = NOW()
                      WHERE id                  = :id
                        AND status              = :status
                        AND current_activations < activation_limit
                        AND deleted_at          IS NULL'
                );
                $result = $stmt->execute([
                    ':domain' => $domain,
                    ':id'     => $license->id,
                    ':status' => 'active',
                ]);

                // Etkilenen satır 0 ise başka bir istek önce davrandı (race condition)
                if ($stmt->rowCount() === 0) {
                    throw new RuntimeException('ACTIVATION_LIMIT_REACHED');
                }
            }
        });

        // --- 5. Güncel modeli yenile ---
        $license->refresh();

        // --- 6. Log ---
        $note = $isReactivation ? 'Mevcut domain ile yeniden aktivasyon.' : 'İlk aktivasyon.';
        $this->writeLog($license->id, 'activated', $domain, $ip, $userAgent, $note);

        // secret_token yalnızca aktivasyon yanıtında döner
        return [
            'license'      => $license,
            'secret_token' => $license->getRawOriginal('secret_token'),
        ];
    }

    // ─── Doğrulama ─────────────────────────────────────────────────────────────

    /**
     * Rutin günlük doğrulama (WordPress eklentisi cron'u tarafından çağrılır).
     * Sadece last_verify_at güncellenir; domain değişmez.
     *
     * @return array{is_active: bool, status: string, admin_message: string|null, valid_until: string|null}
     */
    public function verify(License $license, string $domain, string $ip, string $userAgent): array
    {
        // Domain kilitliyse gelen domain eşleşmeli
        if ($license->isDomainLocked() && !$license->matchesDomain($domain)) {
            $this->writeLog(
                $license->id, 'failed', $domain, $ip, $userAgent,
                "Verify domain uyumsuzluğu: kayıtlı={$license->domain}"
            );

            return [
                'is_active'     => false,
                'status'        => 'domain_mismatch',
                'admin_message' => 'Bu domain bu lisansa kayıtlı değil.',
                'valid_until'   => null,
            ];
        }

        $isActive = $license->isActive();

        if ($isActive) {
            // PDO ile last_verify_at güncelle
            $pdo  = DB::getPdo();
            $stmt = $pdo->prepare(
                'UPDATE licenses
                    SET last_verify_at = NOW(),
                        updated_at     = NOW()
                  WHERE id          = :id
                    AND deleted_at  IS NULL'
            );
            $stmt->execute([':id' => $license->id]);

            $this->writeLog($license->id, 'verified', $domain, $ip, $userAgent);
        } else {
            $reason = $license->isExpired() ? 'Lisans süresi dolmuş.' : 'Lisans aktif değil.';
            $this->writeLog($license->id, 'failed', $domain, $ip, $userAgent, $reason);
        }

        return [
            'is_active'     => $isActive,
            'status'        => $isActive ? 'active' : ($license->isExpired() ? 'expired' : $license->status),
            'admin_message' => $license->admin_message,
            'valid_until'   => $license->valid_until?->toIso8601String(),
        ];
    }

    // ─── Deaktivasyon ──────────────────────────────────────────────────────────

    /**
     * Kullanıcı eklentiyi kaldırdığında veya lisansı taşıdığında çağrılır.
     * Domain kilidini sıfırlar ve aktivasyon sayacını bir azaltır.
     *
     * @throws RuntimeException
     */
    public function deactivate(License $license, string $domain, string $ip, string $userAgent): void
    {
        if (!$license->isDomainLocked()) {
            $this->writeLog($license->id, 'failed', $domain, $ip, $userAgent, 'Zaten aktif değil.');
            throw new RuntimeException('NOT_ACTIVATED');
        }

        if (!$license->matchesDomain($domain)) {
            $this->writeLog(
                $license->id, 'failed', $domain, $ip, $userAgent,
                "Deaktivasyon domain uyumsuzluğu: kayıtlı={$license->domain}"
            );
            throw new RuntimeException('DOMAIN_MISMATCH');
        }

        DB::transaction(function () use ($license) {
            $pdo  = DB::getPdo();
            $stmt = $pdo->prepare(
                'UPDATE licenses
                    SET domain              = NULL,
                        current_activations = GREATEST(current_activations - 1, 0),
                        last_verify_at      = NOW(),
                        updated_at          = NOW()
                  WHERE id         = :id
                    AND deleted_at IS NULL'
            );
            $stmt->execute([':id' => $license->id]);
        });

        $license->refresh();

        $this->writeLog($license->id, 'deactivated', $domain, $ip, $userAgent, 'Lisans başarıyla deaktive edildi.');
    }

    // ─── Private: Audit Log ────────────────────────────────────────────────────

    /**
     * Aktivasyon log kaydını PDO prepared statement ile yazar.
     * Eloquent kullanmama nedeni: transaction dışında çalışabilmesi
     * ve log yazımının hiçbir Eloquent event'ini tetiklememesi.
     */
    private function writeLog(
        int     $licenseId,
        string  $action,
        ?string $domain,
        ?string $ip,
        ?string $userAgent,
        ?string $note = null
    ): void {
        try {
            $pdo  = DB::getPdo();
            $stmt = $pdo->prepare(
                'INSERT INTO license_activation_logs
                    (license_id, action, domain, ip_address, user_agent, note, created_at)
                 VALUES
                    (:license_id, :action, :domain, :ip_address, :user_agent, :note, NOW())'
            );
            $stmt->execute([
                ':license_id' => $licenseId,
                ':action'     => $action,
                ':domain'     => $domain,
                ':ip_address' => $ip,
                ':user_agent' => $userAgent ? substr($userAgent, 0, 512) : null,
                ':note'       => $note,
            ]);
        } catch (\Throwable) {
            // Log hatası ana akışı asla kesmemeli
        }
    }
}
