<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class License extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'license_key',
        'secret_token',
        'customer_name',
        'customer_email',
        'domain',
        'status',
        'activation_limit',
        'current_activations',
        'valid_until',
        'admin_message',
        'last_verify_at',
    ];

    protected $hidden = [
        'secret_token',
    ];

    protected $casts = [
        'valid_until'         => 'datetime',
        'last_verify_at'      => 'datetime',
        'activation_limit'    => 'integer',
        'current_activations' => 'integer',
    ];

    // ─── İlişkiler ─────────────────────────────────────────────────────────────

    public function activationLogs(): HasMany
    {
        return $this->hasMany(LicenseActivationLog::class)->latest('created_at');
    }

    // ─── Scopes ────────────────────────────────────────────────────────────────

    public function scopeActive($query)
    {
        return $query->where('status', 'active')
                     ->where(function ($q) {
                         $q->whereNull('valid_until')
                           ->orWhere('valid_until', '>', now());
                     });
    }

    public function scopeExpired($query)
    {
        return $query->where(function ($q) {
            $q->where('status', 'expired')
              ->orWhere(function ($inner) {
                  $inner->whereNotNull('valid_until')
                        ->where('valid_until', '<=', now());
              });
        });
    }

    // ─── Durum Kontrolleri ─────────────────────────────────────────────────────

    public function isActive(): bool
    {
        return $this->status === 'active'
            && !($this->valid_until && $this->valid_until->isPast());
    }

    public function isSuspended(): bool
    {
        return $this->status === 'suspended';
    }

    public function isExpired(): bool
    {
        return $this->status === 'expired'
            || ($this->valid_until && $this->valid_until->isPast());
    }

    public function hasActivationSlot(): bool
    {
        return $this->current_activations < $this->activation_limit;
    }

    /**
     * Bu lisans daha önce herhangi bir domain'e kilitlenmiş mi?
     */
    public function isDomainLocked(): bool
    {
        return $this->domain !== null && $this->domain !== '';
    }

    /**
     * Verilen domain bu lisansın kilitli olduğu domain ile eşleşiyor mu?
     * Domain karşılaştırması; www. öneki ve case-insensitive normalize edilir.
     */
    public function matchesDomain(string $requestedDomain): bool
    {
        if (!$this->isDomainLocked()) {
            return true; // Henüz kilitlenmemiş
        }

        return $this->normalizeDomain($this->domain)
            === $this->normalizeDomain($requestedDomain);
    }

    // ─── Yardımcı ──────────────────────────────────────────────────────────────

    private function normalizeDomain(string $domain): string
    {
        $domain = strtolower(trim($domain));
        $domain = preg_replace('#^https?://#', '', $domain);
        $domain = ltrim($domain, 'www.');
        return rtrim($domain, '/');
    }
}
