<?php

namespace App\Http\Traits;

use Illuminate\Http\JsonResponse;
use Symfony\Component\HttpFoundation\Response;

/**
 * Standart JSON API yanıt formatı.
 *
 * Her yanıt şu yapıyı taşır:
 * {
 *   "status":    "success" | "error",
 *   "message":   "İnsan okunabilir açıklama",
 *   "data":      { ... } | null,
 *   "timestamp": "2024-01-01T12:00:00+00:00"
 * }
 */
trait ApiResponse
{
    // ─── Başarılı Yanıtlar ─────────────────────────────────────────────────────

    /**
     * Genel başarı yanıtı.
     *
     * @param  mixed       $data    Dönülecek veri (null olabilir)
     * @param  string      $message İnsan okunabilir mesaj
     * @param  int         $code    HTTP durum kodu
     * @param  array       $meta    Ek meta veriler (sayfalama vb.)
     */
    protected function success(
        mixed $data = null,
        string $message = 'İşlem başarılı.',
        int $code = Response::HTTP_OK,
        array $meta = []
    ): JsonResponse {
        $body = [
            'status'    => 'success',
            'message'   => $message,
            'data'      => $data,
            'timestamp' => now()->toIso8601String(),
        ];

        if (!empty($meta)) {
            $body['meta'] = $meta;
        }

        return response()->json($body, $code);
    }

    /**
     * 201 Created — yeni kaynak oluşturuldu.
     */
    protected function created(
        mixed $data = null,
        string $message = 'Kaynak başarıyla oluşturuldu.'
    ): JsonResponse {
        return $this->success($data, $message, Response::HTTP_CREATED);
    }

    /**
     * 204 No Content — veri dönmeksizin başarı.
     * JSON body yok; HTTP spec gereği.
     */
    protected function noContent(): JsonResponse
    {
        return response()->json(null, Response::HTTP_NO_CONTENT);
    }

    // ─── Hata Yanıtları ────────────────────────────────────────────────────────

    /**
     * Genel hata yanıtı.
     *
     * @param  string      $message İnsan okunabilir hata mesajı
     * @param  int         $code    HTTP durum kodu
     * @param  string|null $errorCode  Makine okunabilir hata kodu (ör: LICENSE_EXPIRED)
     * @param  mixed       $errors  Doğrulama hataları veya ek detaylar
     */
    protected function error(
        string $message = 'Bir hata oluştu.',
        int $code = Response::HTTP_BAD_REQUEST,
        ?string $errorCode = null,
        mixed $errors = null
    ): JsonResponse {
        $body = [
            'status'    => 'error',
            'message'   => $message,
            'data'      => null,
            'timestamp' => now()->toIso8601String(),
        ];

        if ($errorCode !== null) {
            $body['error_code'] = $errorCode;
        }

        if ($errors !== null) {
            $body['errors'] = $errors;
        }

        return response()->json($body, $code);
    }

    /**
     * 400 Bad Request
     */
    protected function badRequest(
        string $message = 'Geçersiz istek.',
        mixed $errors = null
    ): JsonResponse {
        return $this->error($message, Response::HTTP_BAD_REQUEST, 'BAD_REQUEST', $errors);
    }

    /**
     * 401 Unauthorized
     */
    protected function unauthorized(
        string $message = 'Yetkilendirme başarısız.',
        string $errorCode = 'UNAUTHORIZED'
    ): JsonResponse {
        return $this->error($message, Response::HTTP_UNAUTHORIZED, $errorCode);
    }

    /**
     * 403 Forbidden
     */
    protected function forbidden(
        string $message = 'Bu işlem için yetkiniz yok.',
        string $errorCode = 'FORBIDDEN'
    ): JsonResponse {
        return $this->error($message, Response::HTTP_FORBIDDEN, $errorCode);
    }

    /**
     * 404 Not Found
     */
    protected function notFound(
        string $message = 'Kaynak bulunamadı.',
        string $errorCode = 'NOT_FOUND'
    ): JsonResponse {
        return $this->error($message, Response::HTTP_NOT_FOUND, $errorCode);
    }

    /**
     * 422 Unprocessable Entity — doğrulama hatası.
     */
    protected function validationError(mixed $errors, string $message = 'Doğrulama hatası.'): JsonResponse
    {
        return $this->error(
            $message,
            Response::HTTP_UNPROCESSABLE_ENTITY,
            'VALIDATION_ERROR',
            $errors
        );
    }

    /**
     * 429 Too Many Requests
     */
    protected function tooManyRequests(string $message = 'Çok fazla istek gönderildi.'): JsonResponse
    {
        return $this->error($message, Response::HTTP_TOO_MANY_REQUESTS, 'RATE_LIMITED');
    }

    /**
     * 500 Internal Server Error
     */
    protected function serverError(string $message = 'Sunucu hatası oluştu.'): JsonResponse
    {
        return $this->error($message, Response::HTTP_INTERNAL_SERVER_ERROR, 'SERVER_ERROR');
    }

    // ─── Domain-Specific Shortcuts ─────────────────────────────────────────────

    /**
     * Lisans durum yanıtları için özelleştirilmiş helper.
     */
    protected function licenseResponse(
        string $licenseStatus,
        string $message,
        mixed $data = null
    ): JsonResponse {
        $httpCode = match ($licenseStatus) {
            'active'    => Response::HTTP_OK,
            'suspended' => Response::HTTP_FORBIDDEN,
            'expired'   => Response::HTTP_PAYMENT_REQUIRED, // 402
            default     => Response::HTTP_BAD_REQUEST,
        };

        $body = [
            'status'          => $licenseStatus === 'active' ? 'success' : 'error',
            'license_status'  => $licenseStatus,
            'message'         => $message,
            'data'            => $data,
            'timestamp'       => now()->toIso8601String(),
        ];

        return response()->json($body, $httpCode);
    }
}
