<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Models\License;
use App\Services\LicenseKeyGenerator;
use App\Services\LicenseService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use RuntimeException;

/**
 * Lisans Yönetim API — v1
 *
 * Tüm endpoint'ler VerifyHmacSignature middleware'i tarafından korunur.
 * Middleware, doğrulanmış License nesnesini $request->attributes('license') ile geçirir.
 *
 * Routes (routes/api.php):
 *   Route::prefix('v1')->middleware('hmac')->group(function () {
 *       Route::post('license/activate',   [LicenseController::class, 'activate']);
 *       Route::post('license/verify',     [LicenseController::class, 'verify']);
 *       Route::post('license/deactivate', [LicenseController::class, 'deactivate']);
 *   });
 *
 *   // Admin endpoint'leri (ayrı auth)
 *   Route::prefix('v1')->middleware('auth:sanctum')->group(function () {
 *       Route::post('admin/licenses',          [LicenseController::class, 'store']);
 *       Route::get('admin/licenses/{key}',     [LicenseController::class, 'show']);
 *   });
 */
class LicenseController extends Controller
{
    use ApiResponse;

    public function __construct(
        private readonly LicenseService      $licenseService,
        private readonly LicenseKeyGenerator $keyGenerator,
    ) {}

    // =========================================================================
    // POST /api/v1/license/activate
    // =========================================================================

    /**
     * İlk aktivasyonu gerçekleştirir.
     *
     * - Domain veritabanına kilitlenir.
     * - Aktivasyon sayacı artırılır.
     * - Yanıtta bir defaya mahsus secret_token döner.
     *
     * Body: { "domain": "example.com" }
     */
    public function activate(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'domain' => 'required|string|max:255|regex:/^[a-zA-Z0-9\-\.]+$/',
        ]);

        if ($validator->fails()) {
            return $this->validationError($validator->errors());
        }

        /** @var License $license */
        $license = $request->attributes->get('license');

        try {
            $result = $this->licenseService->activate(
                license:   $license,
                domain:    $request->input('domain'),
                ip:        $request->ip(),
                userAgent: $request->userAgent() ?? '',
            );
        } catch (RuntimeException $e) {
            return $this->activationErrorResponse($e->getMessage(), $license);
        }

        return $this->created([
            'license_key'         => $result['license']->license_key,
            'secret_token'        => $result['secret_token'],  // Tek seferlik
            'domain'              => $result['license']->domain,
            'customer_name'       => $result['license']->customer_name,
            'current_activations' => $result['license']->current_activations,
            'activation_limit'    => $result['license']->activation_limit,
            'valid_until'         => $result['license']->valid_until?->toIso8601String(),
            'admin_message'       => $result['license']->admin_message,
        ], 'Lisans başarıyla aktive edildi.');
    }

    // =========================================================================
    // POST /api/v1/license/verify
    // =========================================================================

    /**
     * Rutin günlük doğrulama (WordPress cron'u tarafından çağrılır).
     *
     * - Lisans aktif mi?
     * - admin_message var mı?
     * - valid_until ne zaman?
     *
     * Body: { "domain": "example.com" }
     */
    public function verify(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'domain' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return $this->validationError($validator->errors());
        }

        /** @var License $license */
        $license = $request->attributes->get('license');

        $result = $this->licenseService->verify(
            license:   $license,
            domain:    $request->input('domain'),
            ip:        $request->ip(),
            userAgent: $request->userAgent() ?? '',
        );

        if (!$result['is_active']) {
            return $this->licenseResponse(
                licenseStatus: $result['status'],
                message:       $result['admin_message'] ?? $this->statusMessage($result['status']),
                data: [
                    'license_status' => $result['status'],
                    'valid_until'    => $result['valid_until'],
                ]
            );
        }

        return $this->success([
            'license_status' => 'active',
            'valid_until'    => $result['valid_until'],
            'admin_message'  => $result['admin_message'],
        ], 'Lisans doğrulandı.');
    }

    // =========================================================================
    // POST /api/v1/license/deactivate
    // =========================================================================

    /**
     * Kullanıcı eklentiyi kaldırdığında veya lisansı taşıdığında çalışır.
     *
     * - Domain kilidi sıfırlanır.
     * - Aktivasyon sayacı bir azaltılır.
     * - Lisans başka bir domain'e taşınabilir hale gelir.
     *
     * Body: { "domain": "example.com" }
     */
    public function deactivate(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'domain' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return $this->validationError($validator->errors());
        }

        /** @var License $license */
        $license = $request->attributes->get('license');

        try {
            $this->licenseService->deactivate(
                license:   $license,
                domain:    $request->input('domain'),
                ip:        $request->ip(),
                userAgent: $request->userAgent() ?? '',
            );
        } catch (RuntimeException $e) {
            return match ($e->getMessage()) {
                'NOT_ACTIVATED'   => $this->badRequest('Bu lisans henüz aktive edilmemiş.', ['code' => 'NOT_ACTIVATED']),
                'DOMAIN_MISMATCH' => $this->forbidden('Domain bu lisansa kayıtlı değil.', 'DOMAIN_MISMATCH'),
                default           => $this->serverError('Deaktivasyon sırasında hata oluştu.'),
            };
        }

        return $this->success(
            data: [
                'license_key'         => $license->license_key,
                'domain'              => null,
                'current_activations' => $license->fresh()->current_activations,
            ],
            message: 'Lisans başarıyla deaktive edildi. Domain kilidi kaldırıldı.'
        );
    }

    // =========================================================================
    // Admin Endpoint'leri
    // =========================================================================

    /**
     * Yeni lisans oluştur (Admin).
     * Bu endpoint HMAC koruması altında değil; Sanctum/admin auth kullanır.
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'customer_name'    => 'required|string|max:150',
            'customer_email'   => 'required|email|max:255',
            'activation_limit' => 'integer|min:1|max:100',
            'valid_until'      => 'nullable|date|after:now',
            'admin_message'    => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return $this->validationError($validator->errors());
        }

        $license = License::create([
            'license_key'      => $this->keyGenerator->generateUnique(),
            'secret_token'     => $this->keyGenerator->generateSecretToken(),
            'customer_name'    => $request->customer_name,
            'customer_email'   => $request->customer_email,
            'activation_limit' => $request->activation_limit ?? 1,
            'valid_until'      => $request->valid_until,
            'admin_message'    => $request->admin_message,
            'status'           => 'active',
        ]);

        $data                  = $license->toArray();
        $data['secret_token']  = $license->getRawOriginal('secret_token'); // Tek seferlik
        $data['hmac_example']  = $this->buildHmacExample($license);

        return $this->created($data, 'Lisans oluşturuldu. secret_token\'ı güvenli saklayın.');
    }

    /**
     * Lisans detayı getir (Admin).
     */
    public function show(string $key): JsonResponse
    {
        $license = License::with('activationLogs')->where('license_key', $key)->first();

        if (!$license) {
            return $this->notFound('Lisans bulunamadı.');
        }

        return $this->success($license, 'Lisans bilgileri getirildi.');
    }

    // ─── Private Helpers ───────────────────────────────────────────────────────

    private function activationErrorResponse(string $errorCode, License $license): JsonResponse
    {
        return match ($errorCode) {
            'SUSPENDED' => $this->licenseResponse(
                'suspended',
                $license->admin_message ?? 'Lisansınız askıya alınmıştır. Destek ile iletişime geçin.',
            ),
            'EXPIRED' => $this->licenseResponse(
                'expired',
                $license->admin_message ?? 'Lisansınızın süresi dolmuştur. Lütfen yenileyin.',
                ['valid_until' => $license->valid_until?->toIso8601String()]
            ),
            'DOMAIN_MISMATCH' => $this->forbidden(
                'Bu lisans başka bir domain\'e kayıtlıdır.',
                'DOMAIN_MISMATCH'
            ),
            'ACTIVATION_LIMIT_REACHED' => $this->forbidden(
                "Maksimum aktivasyon sayısına ({$license->activation_limit}) ulaşıldı.",
                'ACTIVATION_LIMIT_REACHED'
            ),
            default => $this->badRequest('Aktivasyon gerçekleştirilemedi.', ['code' => $errorCode]),
        };
    }

    private function statusMessage(string $status): string
    {
        return match ($status) {
            'expired'        => 'Lisansınızın süresi dolmuştur.',
            'suspended'      => 'Lisansınız askıya alınmıştır.',
            'domain_mismatch'=> 'Domain bu lisansa kayıtlı değil.',
            default          => 'Lisans geçersiz.',
        };
    }

    /**
     * Yeni oluşturulan lisans için PHP istemci örneği üretir (onboarding yardımcısı).
     */
    private function buildHmacExample(License $license): array
    {
        return [
            'description' => 'WordPress eklentinizde HMAC imzası oluşturmak için örnek PHP kodu',
            'php_snippet' => implode("\n", [
                '$method    = "POST";',
                '$path      = "api/v1/license/verify";',
                '$timestamp = (string) time();',
                '$nonce     = bin2hex(random_bytes(16));',
                '$body      = json_encode(["domain" => home_url()]);',
                '$payload   = implode("\\n", [$method, $path, $timestamp, $nonce, $body]);',
                '$signature = hash_hmac("sha256", $payload, $secretToken);',
            ]),
        ];
    }
}
