<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Models\License;
use App\Services\LicenseKeyGenerator;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

/**
 * Lisans yönetim API controller'ı.
 *
 * Routes (routes/api.php):
 *   POST   /api/licenses              → store()   [Admin]
 *   POST   /api/licenses/verify       → verify()  [HMAC korumalı]
 *   POST   /api/licenses/activate     → activate() [HMAC korumalı]
 *   GET    /api/licenses/{key}        → show()    [Admin]
 */
class LicenseController extends Controller
{
    use ApiResponse;

    public function __construct(
        private readonly LicenseKeyGenerator $generator
    ) {}

    // ─── Admin: Lisans Oluştur ─────────────────────────────────────────────────

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'customer_name'    => 'required|string|max:150',
            'customer_email'   => 'required|email|max:255',
            'domain'           => 'nullable|string|max:255',
            'activation_limit' => 'integer|min:1|max:100',
            'valid_until'      => 'nullable|date|after:now',
            'admin_message'    => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return $this->validationError($validator->errors());
        }

        $license = License::create([
            'license_key'      => $this->generator->generateUnique(),
            'secret_token'     => $this->generator->generateSecretToken(),
            'customer_name'    => $request->customer_name,
            'customer_email'   => $request->customer_email,
            'domain'           => $request->domain,
            'activation_limit' => $request->activation_limit ?? 1,
            'valid_until'      => $request->valid_until,
            'admin_message'    => $request->admin_message,
            'status'           => 'active',
        ]);

        // secret_token sadece oluşturma anında döner, sonra gizlenir
        $responseData = $license->toArray();
        $responseData['secret_token'] = $license->getRawOriginal('secret_token');

        return $this->created($responseData, 'Lisans başarıyla oluşturuldu.');
    }

    // ─── HMAC Korumalı: Lisans Doğrula ────────────────────────────────────────

    /**
     * WordPress eklentisi bu endpoint'i çağırır.
     * VerifyHmacSignature middleware request'e license nesnesini ekler.
     */
    public function verify(Request $request): JsonResponse
    {
        /** @var License $license */
        $license = $request->attributes->get('license');

        if (!$license->isActive()) {
            $status = $license->isExpired() ? 'expired' : $license->status;

            return $this->licenseResponse(
                $status,
                $license->admin_message ?? match ($status) {
                    'expired'   => 'Lisansınızın süresi dolmuştur.',
                    'suspended' => 'Lisansınız askıya alınmıştır. Destek ile iletişime geçin.',
                    default     => 'Lisans geçersiz.',
                }
            );
        }

        // Başarılı doğrulamada son kontrol zamanını güncelle
        $license->update(['last_verify_at' => now()]);

        return $this->licenseResponse('active', 'Lisans geçerli.', [
            'customer_name'       => $license->customer_name,
            'domain'              => $license->domain,
            'valid_until'         => $license->valid_until?->toIso8601String(),
            'current_activations' => $license->current_activations,
            'activation_limit'    => $license->activation_limit,
            'admin_message'       => $license->admin_message,
        ]);
    }

    // ─── HMAC Korumalı: Lisans Aktive Et ──────────────────────────────────────

    public function activate(Request $request): JsonResponse
    {
        /** @var License $license */
        $license = $request->attributes->get('license');

        if (!$license->isActive()) {
            return $this->licenseResponse(
                $license->isExpired() ? 'expired' : $license->status,
                'Aktif olmayan lisans aktive edilemez.'
            );
        }

        if (!$license->hasActivationSlot()) {
            return $this->forbidden(
                "Maksimum aktivasyon sınırına ulaşıldı ({$license->activation_limit}).",
                'ACTIVATION_LIMIT_REACHED'
            );
        }

        $license->increment('current_activations');
        $license->update(['last_verify_at' => now()]);

        return $this->success([
            'current_activations' => $license->fresh()->current_activations,
            'activation_limit'    => $license->activation_limit,
        ], 'Lisans başarıyla aktive edildi.');
    }

    // ─── Admin: Lisans Detay ───────────────────────────────────────────────────

    public function show(string $key): JsonResponse
    {
        $license = License::where('license_key', $key)->first();

        if (!$license) {
            return $this->notFound('Lisans bulunamadı.');
        }

        return $this->success($license, 'Lisans bilgileri getirildi.');
    }
}
