<?php

namespace App\Filament\Widgets;

use App\Models\License;
use App\Models\LicenseActivationLog;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Illuminate\Support\Facades\DB;

class StatsOverviewWidget extends BaseWidget
{
    /**
     * Widget'ın otomatik yenileme süresi (saniye).
     * null = otomatik yenileme yok; canlı sayfa için 30 ayarlayın.
     */
    protected static ?string $pollingInterval = '30s';

    protected static ?int $sort = 1;

    protected function getStats(): array
    {
        // ── Veriler ────────────────────────────────────────────────────────────

        $totalLicenses = License::count();

        $activeLicenses = License::where('status', 'active')
            ->where(function ($q) {
                $q->whereNull('valid_until')
                  ->orWhere('valid_until', '>', now());
            })->count();

        $activeActivations = License::where('status', 'active')
            ->whereNotNull('domain')
            ->where('domain', '!=', '')
            ->count();

        // Son 30 günlük aktivasyon trendi (grafik için)
        $activationTrend = $this->getActivationTrend();

        // Kritik: Süresi dolmuş VEYA 7 gün içinde dolacak
        $criticalCount = License::where(function ($q) {
            $q->where('status', 'expired')
              ->orWhere(function ($inner) {
                  $inner->where('status', 'active')
                        ->whereNotNull('valid_until')
                        ->where('valid_until', '<=', now()->addDays(7));
              });
        })->count();

        // Askıya alınanlar
        $suspendedCount = License::where('status', 'suspended')->count();

        // Son 24 saatte yeni aktivasyon
        $recentActivations = LicenseActivationLog::where('action', 'activated')
            ->where('created_at', '>=', now()->subHours(24))
            ->count();

        // ── Stat Kartları ──────────────────────────────────────────────────────

        return [
            Stat::make('Toplam Lisans', number_format($totalLicenses))
                ->description('Sistemdeki tüm lisans anahtarları')
                ->descriptionIcon('heroicon-m-key')
                ->color('primary')
                ->chart($activationTrend),

            Stat::make('Aktif Lisanslar', number_format($activeLicenses))
                ->description("{$activeActivations} domain'de kullanımda")
                ->descriptionIcon('heroicon-m-check-circle')
                ->color('success')
                ->chart(array_fill(0, 7, $activeLicenses)),

            Stat::make('Kritik Durum', number_format($criticalCount))
                ->description('Süresi dolan veya 7 günde bitecek')
                ->descriptionIcon('heroicon-m-exclamation-triangle')
                ->color($criticalCount > 0 ? 'danger' : 'success'),

            Stat::make('Son 24 Saat', number_format($recentActivations))
                ->description('Yeni aktivasyon / doğrulama')
                ->descriptionIcon('heroicon-m-clock')
                ->color('info'),

            Stat::make('Askıya Alınan', number_format($suspendedCount))
                ->description('Kill-switch uygulanmış lisanslar')
                ->descriptionIcon('heroicon-m-no-symbol')
                ->color($suspendedCount > 0 ? 'warning' : 'gray'),
        ];
    }

    /**
     * Son 7 günlük günlük aktivasyon sayıları (sparkline için)
     */
    private function getActivationTrend(): array
    {
        $data = LicenseActivationLog::select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('COUNT(*) as count')
            )
            ->where('action', 'activated')
            ->where('created_at', '>=', now()->subDays(7))
            ->groupBy('date')
            ->orderBy('date')
            ->pluck('count', 'date')
            ->toArray();

        // 7 günlük tam diziyi doldur (eksik günler 0)
        $trend = [];
        for ($i = 6; $i >= 0; $i--) {
            $date    = now()->subDays($i)->toDateString();
            $trend[] = $data[$date] ?? 0;
        }

        return $trend;
    }
}
