<?php

namespace App\Filament\Widgets;

use App\Models\LicenseActivationLog;
use App\Models\License;
use Filament\Widgets\ChartWidget;
use Illuminate\Support\Facades\DB;

class LicenseChartWidget extends ChartWidget
{
    protected static ?string $heading    = 'Aktivasyon Trendi (Son 30 Gün)';
    protected static ?int    $sort       = 2;
    protected int | string | array $columnSpan = 'full';

    // Filtre seçenekleri
    public ?string $filter = '30';

    protected function getFilters(): ?array
    {
        return [
            '7'  => 'Son 7 Gün',
            '30' => 'Son 30 Gün',
            '90' => 'Son 90 Gün',
        ];
    }

    protected function getData(): array
    {
        $days = (int) ($this->filter ?? 30);

        // Günlük aktivasyon ve doğrulama sayıları
        $activations = LicenseActivationLog::select(
                DB::raw('DATE(created_at) as date'),
                DB::raw("SUM(CASE WHEN action = 'activated' THEN 1 ELSE 0 END) as activated"),
                DB::raw("SUM(CASE WHEN action = 'verified' THEN 1 ELSE 0 END) as verified"),
                DB::raw("SUM(CASE WHEN action = 'failed' THEN 1 ELSE 0 END) as failed")
            )
            ->where('created_at', '>=', now()->subDays($days))
            ->groupBy('date')
            ->orderBy('date')
            ->get()
            ->keyBy('date');

        // Tam tarih aralığını oluştur
        $labels      = [];
        $activated   = [];
        $verified    = [];
        $failed      = [];

        for ($i = $days - 1; $i >= 0; $i--) {
            $date      = now()->subDays($i)->toDateString();
            $labels[]  = now()->subDays($i)->format('d M');
            $row       = $activations->get($date);

            $activated[] = $row?->activated ?? 0;
            $verified[]  = $row?->verified  ?? 0;
            $failed[]    = $row?->failed    ?? 0;
        }

        return [
            'datasets' => [
                [
                    'label'           => 'Aktivasyon',
                    'data'            => $activated,
                    'borderColor'     => '#10b981',
                    'backgroundColor' => 'rgba(16, 185, 129, 0.08)',
                    'fill'            => true,
                    'tension'         => 0.4,
                ],
                [
                    'label'           => 'Doğrulama',
                    'data'            => $verified,
                    'borderColor'     => '#6366f1',
                    'backgroundColor' => 'rgba(99, 102, 241, 0.08)',
                    'fill'            => true,
                    'tension'         => 0.4,
                ],
                [
                    'label'           => 'Başarısız',
                    'data'            => $failed,
                    'borderColor'     => '#f43f5e',
                    'backgroundColor' => 'rgba(244, 63, 94, 0.08)',
                    'fill'            => true,
                    'tension'         => 0.4,
                ],
            ],
            'labels' => $labels,
        ];
    }

    protected function getType(): string
    {
        return 'line';
    }
}
