# WordPress Lisans Yönetim Sistemi — Aşama 1

PHP 8.2 + Laravel tabanlı, HMAC-SHA256 korumalı lisans doğrulama API'si.

---

## Proje Yapısı

```
app/
├── Http/
│   ├── Controllers/Api/
│   │   └── LicenseController.php     # API endpoint'leri
│   ├── Middleware/
│   │   └── VerifyHmacSignature.php   # HMAC doğrulama + replay attack koruması
│   └── Traits/
│       └── ApiResponse.php           # Standart JSON yanıt formatı
├── Models/
│   └── License.php                   # Eloquent model
└── Services/
    └── LicenseKeyGenerator.php       # Kriptografik anahtar üretici

database/migrations/
└── ..._create_licenses_table.php     # Veritabanı şeması
```

---

## Kurulum

```bash
# Migration çalıştır
php artisan migrate

# Middleware'i kaydet (bootstrap/app.php veya Kernel.php)
->withMiddleware(function (Middleware $middleware) {
    $middleware->alias([
        'hmac' => \App\Http\Middleware\VerifyHmacSignature::class,
    ]);
})
```

### Route Tanımları (`routes/api.php`)

```php
use App\Http\Controllers\Api\LicenseController;

// Admin endpoint'leri (ayrı bir admin auth middleware eklenebilir)
Route::post('/licenses', [LicenseController::class, 'store']);
Route::get('/licenses/{key}', [LicenseController::class, 'show']);

// HMAC korumalı endpoint'ler
Route::middleware('hmac')->group(function () {
    Route::post('/licenses/verify',   [LicenseController::class, 'verify']);
    Route::post('/licenses/activate', [LicenseController::class, 'activate']);
});
```

---

## API Kullanımı

### 1. Lisans Oluştur (Admin)

```http
POST /api/licenses
Content-Type: application/json

{
  "customer_name": "Ahmet Yılmaz",
  "customer_email": "ahmet@example.com",
  "domain": "example.com",
  "activation_limit": 3,
  "valid_until": "2025-12-31T23:59:59Z"
}
```

**Yanıt:**
```json
{
  "status": "success",
  "message": "Lisans başarıyla oluşturuldu.",
  "data": {
    "license_key": "A3K9-MXPQ-7BNR-2TVW",
    "secret_token": "a1b2c3...64hex...",
    "customer_name": "Ahmet Yılmaz"
  },
  "timestamp": "2024-01-01T12:00:00+00:00"
}
```

> ⚠️ `secret_token` yalnızca oluşturma anında döner. Güvenli şekilde saklayın.

---

### 2. HMAC İmzalı İstek Oluşturma (WordPress Eklentisi)

```php
$method    = 'POST';
$path      = 'api/licenses/verify';
$timestamp = (string) time();
$nonce     = bin2hex(random_bytes(16));  // 32 karakter hex
$body      = json_encode([]);            // Boş body veya JSON

// Payload: her satır \n ile ayrılır
$payload = implode("\n", [$method, $path, $timestamp, $nonce, $body]);

// İmza
$signature = hash_hmac('sha256', $payload, $secretToken);

// Headers
$headers = [
    'X-License-Key' => 'A3K9-MXPQ-7BNR-2TVW',
    'X-Signature'   => $signature,
    'X-Timestamp'   => $timestamp,
    'X-Nonce'       => $nonce,
    'Content-Type'  => 'application/json',
];
```

---

### 3. Lisans Doğrula

```http
POST /api/licenses/verify
X-License-Key: A3K9-MXPQ-7BNR-2TVW
X-Signature: <hmac-sha256-hex>
X-Timestamp: 1704067200
X-Nonce: <32-karakter-hex>
```

**Başarılı Yanıt (200):**
```json
{
  "status": "success",
  "license_status": "active",
  "message": "Lisans geçerli.",
  "data": {
    "customer_name": "Ahmet Yılmaz",
    "domain": "example.com",
    "valid_until": "2025-12-31T23:59:59+00:00",
    "current_activations": 1,
    "activation_limit": 3
  },
  "timestamp": "2024-01-01T12:00:00+00:00"
}
```

**Süresi Dolmuş (402):**
```json
{
  "status": "error",
  "license_status": "expired",
  "message": "Lisansınızın süresi dolmuştur.",
  "data": null,
  "timestamp": "..."
}
```

---

## Güvenlik Mimarisi

| Katman | Mekanizma | Açıklama |
|--------|-----------|----------|
| **İmza** | HMAC-SHA256 | Her istek kriptografik olarak imzalanır |
| **Replay Attack** | Timestamp ±5dk | Eski istekler reddedilir |
| **Replay Attack** | Nonce + Cache | Aynı istek tekrar kullanılamaz |
| **Timing Attack** | `hash_equals()` | Sabit süreli karşılaştırma |
| **Key Entropy** | CSPRNG + Rejection Sampling | Bias'sız ~78 bit entropi |
| **Token Gizliliği** | `$hidden` + tek seferli açıklama | Secret token API'de gizlenir |

---

## LicenseKeyGenerator Detayları

- **Alfabe:** `0-9`, `A-Z` → `O` ve `I` çıkarıldı (34 karakter)
- **Format:** `XXXX-XXXX-XXXX-XXXX` (16 anlamlı karakter)
- **Entropi:** ≈78 bit (kaba kuvvet saldırısına karşı yeterli)
- **Yöntem:** `random_bytes()` + rejection sampling (modulo bias yok)
- **Çakışma:** Üretimde DB kontrolü ile benzersizlik garantisi
