# 🚀 LMS Kurulum Rehberi

Adım adım kurulum — sıfırdan production'a kadar.

---

## Gereksinimler

| Araç | Sürüm |
|------|-------|
| PHP | 8.2+ |
| Composer | 2.x |
| MySQL / MariaDB | 8.0+ / 10.6+ |
| Node.js (opsiyonel) | 18+ |

> **SQLite ile hızlı başlangıç** için MySQL gerekmez — aşağıda belirtildi.

---

## 1. Bağımlılıkları Yükle

```bash
composer install
```

---

## 2. Ortam Dosyasını Hazırla

```bash
cp .env.example .env
php artisan key:generate
```

`.env` dosyasını açıp düzenleyin:

### MySQL için:
```env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=lms_db
DB_USERNAME=root
DB_PASSWORD=your_password

APP_URL=https://your-domain.com
```

### SQLite için (geliştirme):
```env
DB_CONNECTION=sqlite
# DB_DATABASE satırını tamamen sil veya yorum yap
```
SQLite dosyasını oluştur:
```bash
touch database/database.sqlite
```

---

## 3. Veritabanını Oluştur (MySQL)

```sql
CREATE DATABASE lms_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

---

## 4. Migration ve Seed

```bash
# Tabloları oluştur
php artisan migrate

# İlk admin kullanıcısını oluştur (admin@lms.local / password)
php artisan db:seed
```

---

## 5. Storage Link

```bash
php artisan storage:link
```

---

## 6. İzinleri Ayarla (Linux/macOS)

```bash
chmod -R 775 storage bootstrap/cache
chown -R www-data:www-data storage bootstrap/cache  # Apache için
# veya
chown -R nginx:nginx storage bootstrap/cache         # Nginx için
```

---

## 7. Admin Kullanıcısı Oluştur

Seeder çalıştırıldıysa `admin@lms.local / password` hazır.

Manuel oluşturmak için:
```bash
php artisan make:filament-user
# veya
php artisan tinker
>>> App\Models\User::create(['name'=>'Admin','email'=>'admin@example.com','password'=>bcrypt('guclu_sifre')])
```

---

## 8. Uygulamayı Başlat

### Geliştirme:
```bash
php artisan serve
# → http://localhost:8000/admin
```

### Production (Nginx örneği):
```nginx
server {
    listen 80;
    server_name your-domain.com;
    root /var/www/lms/public;
    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }
}
```

---

## 9. Cache Ayarı (Production)

HMAC replay koruması için `Cache` kullanılır.
Varsayılan: **database** (çalışır ama yavaş).
Production için **Redis** önerilir:

```env
CACHE_STORE=redis
REDIS_HOST=127.0.0.1
REDIS_PORT=6379
```

```bash
# Production cache optimize
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

---

## 10. Zamanlanmış Görevler (Cron)

Süresi dolmuş lisansları otomatik işaretle:

```bash
# crontab -e
* * * * * cd /var/www/lms && php artisan schedule:run >> /dev/null 2>&1
```

---

## Panel Adresi

```
https://your-domain.com/admin
```

---

## API Endpoint Özeti

| Yöntem | Endpoint | Koruma | Açıklama |
|--------|----------|--------|----------|
| POST | `/api/v1/license/activate` | HMAC | Domain kilitleme + aktivasyon |
| POST | `/api/v1/license/verify` | HMAC | Günlük doğrulama |
| POST | `/api/v1/license/deactivate` | HMAC | Domain kilidi kaldır |
| POST | `/api/v1/admin/licenses` | Sanctum | Yeni lisans oluştur |
| GET | `/api/v1/admin/licenses/{key}` | Sanctum | Lisans detayı |

---

## HMAC İmza Formatı (WordPress Eklentisi)

```php
$payload = implode("\n", [
    strtoupper($method),   // POST
    $path,                 // api/v1/license/verify
    $timestamp,            // time()
    $nonce,                // bin2hex(random_bytes(16))
    $jsonBody,             // json_encode(['domain' => 'example.com'])
]);
$signature = hash_hmac('sha256', $payload, $secretToken);
```

**Zorunlu Headers:**
```
X-License-Key: XXXX-XXXX-XXXX-XXXX
X-Signature:   <sha256-hex>
X-Timestamp:   <unix-timestamp>
X-Nonce:       <min-16-karakter-hex>
Content-Type:  application/json
```

---

## Sorun Giderme

### `php artisan key:generate` sonrası `.env`'de APP_KEY hâlâ boş
```bash
php artisan config:clear
cat .env | grep APP_KEY
```

### Migration hatası: `Access denied`
`.env` içindeki `DB_USERNAME` / `DB_PASSWORD` kontrol edin.

### Filament paneli açılmıyor
```bash
php artisan filament:upgrade
php artisan optimize:clear
```

### `storage` izin hatası
```bash
chmod -R 777 storage bootstrap/cache
```
